/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      kitegi <kitegi@users.noreply.github.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN fnmsub

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/fnmsub.h>

static int
test_simde_mm512_fnmsub_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde_float32 c[16];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(    38.95), SIMDE_FLOAT32_C(   -13.06), SIMDE_FLOAT32_C(    27.88), SIMDE_FLOAT32_C(    62.28),
        SIMDE_FLOAT32_C(    66.10), SIMDE_FLOAT32_C(   -82.38), SIMDE_FLOAT32_C(     5.33), SIMDE_FLOAT32_C(    59.08),
        SIMDE_FLOAT32_C(    11.50), SIMDE_FLOAT32_C(     2.93), SIMDE_FLOAT32_C(    86.03), SIMDE_FLOAT32_C(   -85.53),
        SIMDE_FLOAT32_C(    41.10), SIMDE_FLOAT32_C(   -96.18), SIMDE_FLOAT32_C(    60.27), SIMDE_FLOAT32_C(    88.72) },
      { SIMDE_FLOAT32_C(    84.81), SIMDE_FLOAT32_C(    87.90), SIMDE_FLOAT32_C(    94.42), SIMDE_FLOAT32_C(   -72.92),
        SIMDE_FLOAT32_C(   -94.27), SIMDE_FLOAT32_C(    82.63), SIMDE_FLOAT32_C(    39.45), SIMDE_FLOAT32_C(    66.67),
        SIMDE_FLOAT32_C(    16.72), SIMDE_FLOAT32_C(     0.77), SIMDE_FLOAT32_C(   -40.95), SIMDE_FLOAT32_C(    62.84),
        SIMDE_FLOAT32_C(   -22.56), SIMDE_FLOAT32_C(   -32.15), SIMDE_FLOAT32_C(    60.85), SIMDE_FLOAT32_C(   -83.61) },
      { SIMDE_FLOAT32_C(    54.78), SIMDE_FLOAT32_C(   -11.27), SIMDE_FLOAT32_C(    78.67), SIMDE_FLOAT32_C(    20.88),
        SIMDE_FLOAT32_C(     6.35), SIMDE_FLOAT32_C(   -16.00), SIMDE_FLOAT32_C(   -20.03), SIMDE_FLOAT32_C(   -82.16),
        SIMDE_FLOAT32_C(    86.93), SIMDE_FLOAT32_C(   -34.00), SIMDE_FLOAT32_C(   -67.68), SIMDE_FLOAT32_C(    28.02),
        SIMDE_FLOAT32_C(   -30.18), SIMDE_FLOAT32_C(    92.59), SIMDE_FLOAT32_C(    16.74), SIMDE_FLOAT32_C(   -45.37) },
      { SIMDE_FLOAT32_C( -3358.13), SIMDE_FLOAT32_C(  1159.24), SIMDE_FLOAT32_C( -2711.10), SIMDE_FLOAT32_C(  4520.58),
        SIMDE_FLOAT32_C(  6224.90), SIMDE_FLOAT32_C(  6823.06), SIMDE_FLOAT32_C(  -190.24), SIMDE_FLOAT32_C( -3856.70),
        SIMDE_FLOAT32_C(  -279.21), SIMDE_FLOAT32_C(    31.74), SIMDE_FLOAT32_C(  3590.61), SIMDE_FLOAT32_C(  5346.69),
        SIMDE_FLOAT32_C(   957.40), SIMDE_FLOAT32_C( -3184.78), SIMDE_FLOAT32_C( -3684.17), SIMDE_FLOAT32_C(  7463.25) } },
    { { SIMDE_FLOAT32_C(    80.49), SIMDE_FLOAT32_C(    11.16), SIMDE_FLOAT32_C(   -18.30), SIMDE_FLOAT32_C(    86.22),
        SIMDE_FLOAT32_C(    -6.21), SIMDE_FLOAT32_C(   -78.85), SIMDE_FLOAT32_C(    52.89), SIMDE_FLOAT32_C(   -89.49),
        SIMDE_FLOAT32_C(    21.92), SIMDE_FLOAT32_C(   -88.06), SIMDE_FLOAT32_C(    73.35), SIMDE_FLOAT32_C(    99.36),
        SIMDE_FLOAT32_C(   -20.21), SIMDE_FLOAT32_C(    34.20), SIMDE_FLOAT32_C(   -84.24), SIMDE_FLOAT32_C(   -65.43) },
      { SIMDE_FLOAT32_C(   -77.07), SIMDE_FLOAT32_C(    94.43), SIMDE_FLOAT32_C(    55.45), SIMDE_FLOAT32_C(    29.28),
        SIMDE_FLOAT32_C(   -21.57), SIMDE_FLOAT32_C(   -64.58), SIMDE_FLOAT32_C(    47.12), SIMDE_FLOAT32_C(   -34.64),
        SIMDE_FLOAT32_C(     1.42), SIMDE_FLOAT32_C(    79.44), SIMDE_FLOAT32_C(    93.38), SIMDE_FLOAT32_C(    71.24),
        SIMDE_FLOAT32_C(    72.03), SIMDE_FLOAT32_C(    10.12), SIMDE_FLOAT32_C(   -74.14), SIMDE_FLOAT32_C(    52.52) },
      { SIMDE_FLOAT32_C(   -78.71), SIMDE_FLOAT32_C(     7.57), SIMDE_FLOAT32_C(    38.73), SIMDE_FLOAT32_C(    15.08),
        SIMDE_FLOAT32_C(    28.72), SIMDE_FLOAT32_C(    -8.38), SIMDE_FLOAT32_C(    25.59), SIMDE_FLOAT32_C(   -49.36),
        SIMDE_FLOAT32_C(     3.56), SIMDE_FLOAT32_C(    -1.06), SIMDE_FLOAT32_C(   -50.00), SIMDE_FLOAT32_C(    83.35),
        SIMDE_FLOAT32_C(   -66.86), SIMDE_FLOAT32_C(   -34.24), SIMDE_FLOAT32_C(   -82.08), SIMDE_FLOAT32_C(   -43.92) },
      { SIMDE_FLOAT32_C(  6282.07), SIMDE_FLOAT32_C( -1061.41), SIMDE_FLOAT32_C(   976.01), SIMDE_FLOAT32_C( -2539.60),
        SIMDE_FLOAT32_C(  -162.67), SIMDE_FLOAT32_C( -5083.75), SIMDE_FLOAT32_C( -2517.77), SIMDE_FLOAT32_C( -3050.57),
        SIMDE_FLOAT32_C(   -34.69), SIMDE_FLOAT32_C(  6996.55), SIMDE_FLOAT32_C( -6799.42), SIMDE_FLOAT32_C( -7161.76),
        SIMDE_FLOAT32_C(  1522.59), SIMDE_FLOAT32_C(  -311.86), SIMDE_FLOAT32_C( -6163.47), SIMDE_FLOAT32_C(  3480.30) } },
    { { SIMDE_FLOAT32_C(   -39.81), SIMDE_FLOAT32_C(    73.37), SIMDE_FLOAT32_C(    85.36), SIMDE_FLOAT32_C(    38.62),
        SIMDE_FLOAT32_C(   -91.20), SIMDE_FLOAT32_C(    32.48), SIMDE_FLOAT32_C(   -96.03), SIMDE_FLOAT32_C(    10.22),
        SIMDE_FLOAT32_C(    11.92), SIMDE_FLOAT32_C(    97.35), SIMDE_FLOAT32_C(   -18.55), SIMDE_FLOAT32_C(   -16.05),
        SIMDE_FLOAT32_C(     7.48), SIMDE_FLOAT32_C(     7.32), SIMDE_FLOAT32_C(   -63.53), SIMDE_FLOAT32_C(    28.76) },
      { SIMDE_FLOAT32_C(   -85.12), SIMDE_FLOAT32_C(    75.20), SIMDE_FLOAT32_C(   -56.16), SIMDE_FLOAT32_C(    43.60),
        SIMDE_FLOAT32_C(   -33.18), SIMDE_FLOAT32_C(    69.43), SIMDE_FLOAT32_C(    94.24), SIMDE_FLOAT32_C(    70.38),
        SIMDE_FLOAT32_C(   -31.63), SIMDE_FLOAT32_C(   -55.76), SIMDE_FLOAT32_C(    53.73), SIMDE_FLOAT32_C(     1.51),
        SIMDE_FLOAT32_C(    10.00), SIMDE_FLOAT32_C(    71.65), SIMDE_FLOAT32_C(    57.59), SIMDE_FLOAT32_C(    70.19) },
      { SIMDE_FLOAT32_C(    45.02), SIMDE_FLOAT32_C(    42.95), SIMDE_FLOAT32_C(     8.80), SIMDE_FLOAT32_C(    53.81),
        SIMDE_FLOAT32_C(   -24.57), SIMDE_FLOAT32_C(    12.78), SIMDE_FLOAT32_C(   -35.97), SIMDE_FLOAT32_C(    87.34),
        SIMDE_FLOAT32_C(    10.13), SIMDE_FLOAT32_C(    45.48), SIMDE_FLOAT32_C(   -28.70), SIMDE_FLOAT32_C(   -82.39),
        SIMDE_FLOAT32_C(   -47.20), SIMDE_FLOAT32_C(     7.76), SIMDE_FLOAT32_C(    46.37), SIMDE_FLOAT32_C(   -32.32) },
      { SIMDE_FLOAT32_C( -3433.65), SIMDE_FLOAT32_C( -5560.37), SIMDE_FLOAT32_C(  4785.02), SIMDE_FLOAT32_C( -1737.64),
        SIMDE_FLOAT32_C( -3001.45), SIMDE_FLOAT32_C( -2267.87), SIMDE_FLOAT32_C(  9085.84), SIMDE_FLOAT32_C(  -806.62),
        SIMDE_FLOAT32_C(   366.90), SIMDE_FLOAT32_C(  5382.76), SIMDE_FLOAT32_C(  1025.39), SIMDE_FLOAT32_C(   106.63),
        SIMDE_FLOAT32_C(   -27.60), SIMDE_FLOAT32_C(  -532.24), SIMDE_FLOAT32_C(  3612.32), SIMDE_FLOAT32_C( -1986.34) } },
    { { SIMDE_FLOAT32_C(   -17.04), SIMDE_FLOAT32_C(    90.21), SIMDE_FLOAT32_C(   -88.72), SIMDE_FLOAT32_C(    49.79),
        SIMDE_FLOAT32_C(    59.64), SIMDE_FLOAT32_C(   -94.48), SIMDE_FLOAT32_C(    20.17), SIMDE_FLOAT32_C(   -71.99),
        SIMDE_FLOAT32_C(   -50.24), SIMDE_FLOAT32_C(   -26.10), SIMDE_FLOAT32_C(    29.52), SIMDE_FLOAT32_C(    59.76),
        SIMDE_FLOAT32_C(   -54.46), SIMDE_FLOAT32_C(   -12.89), SIMDE_FLOAT32_C(    29.94), SIMDE_FLOAT32_C(    90.56) },
      { SIMDE_FLOAT32_C(   -69.94), SIMDE_FLOAT32_C(   -61.25), SIMDE_FLOAT32_C(    44.38), SIMDE_FLOAT32_C(     5.48),
        SIMDE_FLOAT32_C(    51.53), SIMDE_FLOAT32_C(   -91.59), SIMDE_FLOAT32_C(    -7.17), SIMDE_FLOAT32_C(   -38.34),
        SIMDE_FLOAT32_C(    53.89), SIMDE_FLOAT32_C(    64.12), SIMDE_FLOAT32_C(   -20.73), SIMDE_FLOAT32_C(   -93.31),
        SIMDE_FLOAT32_C(   -28.11), SIMDE_FLOAT32_C(   -74.36), SIMDE_FLOAT32_C(   -25.63), SIMDE_FLOAT32_C(    54.85) },
      { SIMDE_FLOAT32_C(   -84.15), SIMDE_FLOAT32_C(   -14.35), SIMDE_FLOAT32_C(     4.64), SIMDE_FLOAT32_C(    75.49),
        SIMDE_FLOAT32_C(    -8.83), SIMDE_FLOAT32_C(   -75.19), SIMDE_FLOAT32_C(   -96.50), SIMDE_FLOAT32_C(    40.92),
        SIMDE_FLOAT32_C(    -1.30), SIMDE_FLOAT32_C(    33.03), SIMDE_FLOAT32_C(     0.68), SIMDE_FLOAT32_C(    44.25),
        SIMDE_FLOAT32_C(   -79.86), SIMDE_FLOAT32_C(   -69.37), SIMDE_FLOAT32_C(    34.81), SIMDE_FLOAT32_C(   -49.80) },
      { SIMDE_FLOAT32_C( -1107.63), SIMDE_FLOAT32_C(  5539.71), SIMDE_FLOAT32_C(  3932.75), SIMDE_FLOAT32_C(  -348.34),
        SIMDE_FLOAT32_C( -3064.42), SIMDE_FLOAT32_C( -8578.23), SIMDE_FLOAT32_C(   241.12), SIMDE_FLOAT32_C( -2801.02),
        SIMDE_FLOAT32_C(  2708.73), SIMDE_FLOAT32_C(  1640.50), SIMDE_FLOAT32_C(   611.27), SIMDE_FLOAT32_C(  5531.96),
        SIMDE_FLOAT32_C( -1451.01), SIMDE_FLOAT32_C(  -889.13), SIMDE_FLOAT32_C(   732.55), SIMDE_FLOAT32_C( -4917.42) } },
    { { SIMDE_FLOAT32_C(   -30.63), SIMDE_FLOAT32_C(   -20.81), SIMDE_FLOAT32_C(    55.68), SIMDE_FLOAT32_C(   -79.10),
        SIMDE_FLOAT32_C(   -12.41), SIMDE_FLOAT32_C(   -51.49), SIMDE_FLOAT32_C(   -17.44), SIMDE_FLOAT32_C(   -58.52),
        SIMDE_FLOAT32_C(   -87.36), SIMDE_FLOAT32_C(    61.82), SIMDE_FLOAT32_C(   -51.83), SIMDE_FLOAT32_C(   -15.47),
        SIMDE_FLOAT32_C(    87.46), SIMDE_FLOAT32_C(    22.53), SIMDE_FLOAT32_C(   -60.62), SIMDE_FLOAT32_C(   -96.69) },
      { SIMDE_FLOAT32_C(   -91.82), SIMDE_FLOAT32_C(    44.02), SIMDE_FLOAT32_C(    78.80), SIMDE_FLOAT32_C(    -0.65),
        SIMDE_FLOAT32_C(    68.82), SIMDE_FLOAT32_C(    82.31), SIMDE_FLOAT32_C(   -59.73), SIMDE_FLOAT32_C(   -32.48),
        SIMDE_FLOAT32_C(    15.33), SIMDE_FLOAT32_C(    40.95), SIMDE_FLOAT32_C(   -88.23), SIMDE_FLOAT32_C(    35.47),
        SIMDE_FLOAT32_C(    71.58), SIMDE_FLOAT32_C(    46.58), SIMDE_FLOAT32_C(    85.67), SIMDE_FLOAT32_C(   -59.05) },
      { SIMDE_FLOAT32_C(   -74.23), SIMDE_FLOAT32_C(    41.35), SIMDE_FLOAT32_C(   -38.15), SIMDE_FLOAT32_C(    13.36),
        SIMDE_FLOAT32_C(    89.87), SIMDE_FLOAT32_C(    44.41), SIMDE_FLOAT32_C(    54.84), SIMDE_FLOAT32_C(   -97.49),
        SIMDE_FLOAT32_C(     6.23), SIMDE_FLOAT32_C(   -97.00), SIMDE_FLOAT32_C(   -12.97), SIMDE_FLOAT32_C(    -6.30),
        SIMDE_FLOAT32_C(    25.54), SIMDE_FLOAT32_C(    26.41), SIMDE_FLOAT32_C(    -2.99), SIMDE_FLOAT32_C(    33.72) },
      { SIMDE_FLOAT32_C( -2738.22), SIMDE_FLOAT32_C(   874.71), SIMDE_FLOAT32_C( -4349.43), SIMDE_FLOAT32_C(   -64.77),
        SIMDE_FLOAT32_C(   764.19), SIMDE_FLOAT32_C(  4193.73), SIMDE_FLOAT32_C( -1096.53), SIMDE_FLOAT32_C( -1803.24),
        SIMDE_FLOAT32_C(  1333.00), SIMDE_FLOAT32_C( -2434.53), SIMDE_FLOAT32_C( -4559.99), SIMDE_FLOAT32_C(   555.02),
        SIMDE_FLOAT32_C( -6285.93), SIMDE_FLOAT32_C( -1075.86), SIMDE_FLOAT32_C(  5196.31), SIMDE_FLOAT32_C( -5743.26) } },
    { { SIMDE_FLOAT32_C(   -29.57), SIMDE_FLOAT32_C(   -24.19), SIMDE_FLOAT32_C(   -66.94), SIMDE_FLOAT32_C(   -60.75),
        SIMDE_FLOAT32_C(   -41.88), SIMDE_FLOAT32_C(   -26.67), SIMDE_FLOAT32_C(     6.77), SIMDE_FLOAT32_C(    73.45),
        SIMDE_FLOAT32_C(   -85.71), SIMDE_FLOAT32_C(    18.54), SIMDE_FLOAT32_C(     8.92), SIMDE_FLOAT32_C(    85.87),
        SIMDE_FLOAT32_C(   -34.88), SIMDE_FLOAT32_C(    -5.41), SIMDE_FLOAT32_C(   -73.18), SIMDE_FLOAT32_C(    -9.11) },
      { SIMDE_FLOAT32_C(   -64.05), SIMDE_FLOAT32_C(   -11.32), SIMDE_FLOAT32_C(   -95.75), SIMDE_FLOAT32_C(   -74.18),
        SIMDE_FLOAT32_C(   -66.91), SIMDE_FLOAT32_C(    59.08), SIMDE_FLOAT32_C(   -71.68), SIMDE_FLOAT32_C(    39.32),
        SIMDE_FLOAT32_C(    62.09), SIMDE_FLOAT32_C(    15.35), SIMDE_FLOAT32_C(   -66.98), SIMDE_FLOAT32_C(   -12.37),
        SIMDE_FLOAT32_C(   -58.24), SIMDE_FLOAT32_C(    30.03), SIMDE_FLOAT32_C(   -78.66), SIMDE_FLOAT32_C(    12.19) },
      { SIMDE_FLOAT32_C(   -94.16), SIMDE_FLOAT32_C(   -45.59), SIMDE_FLOAT32_C(    51.44), SIMDE_FLOAT32_C(   -36.04),
        SIMDE_FLOAT32_C(    27.74), SIMDE_FLOAT32_C(   -41.79), SIMDE_FLOAT32_C(   -62.59), SIMDE_FLOAT32_C(    42.03),
        SIMDE_FLOAT32_C(    76.75), SIMDE_FLOAT32_C(    46.33), SIMDE_FLOAT32_C(    27.90), SIMDE_FLOAT32_C(   -58.12),
        SIMDE_FLOAT32_C(   -59.08), SIMDE_FLOAT32_C(    54.72), SIMDE_FLOAT32_C(    32.77), SIMDE_FLOAT32_C(   -23.13) },
      { SIMDE_FLOAT32_C( -1799.80), SIMDE_FLOAT32_C(  -228.24), SIMDE_FLOAT32_C( -6460.95), SIMDE_FLOAT32_C( -4470.40),
        SIMDE_FLOAT32_C( -2829.93), SIMDE_FLOAT32_C(  1617.45), SIMDE_FLOAT32_C(   547.86), SIMDE_FLOAT32_C( -2930.08),
        SIMDE_FLOAT32_C(  5244.98), SIMDE_FLOAT32_C(  -330.92), SIMDE_FLOAT32_C(   569.56), SIMDE_FLOAT32_C(  1120.33),
        SIMDE_FLOAT32_C( -1972.33), SIMDE_FLOAT32_C(   107.74), SIMDE_FLOAT32_C( -5789.11), SIMDE_FLOAT32_C(   134.18) } },
    { { SIMDE_FLOAT32_C(   -56.60), SIMDE_FLOAT32_C(    37.01), SIMDE_FLOAT32_C(     2.69), SIMDE_FLOAT32_C(   -23.52),
        SIMDE_FLOAT32_C(    -3.90), SIMDE_FLOAT32_C(    31.01), SIMDE_FLOAT32_C(   -84.20), SIMDE_FLOAT32_C(   -41.81),
        SIMDE_FLOAT32_C(   -53.64), SIMDE_FLOAT32_C(   -51.18), SIMDE_FLOAT32_C(    45.81), SIMDE_FLOAT32_C(   -11.87),
        SIMDE_FLOAT32_C(    78.85), SIMDE_FLOAT32_C(    67.16), SIMDE_FLOAT32_C(   -99.68), SIMDE_FLOAT32_C(    84.69) },
      { SIMDE_FLOAT32_C(   -78.44), SIMDE_FLOAT32_C(    51.75), SIMDE_FLOAT32_C(   -51.36), SIMDE_FLOAT32_C(    49.31),
        SIMDE_FLOAT32_C(   -90.04), SIMDE_FLOAT32_C(   -13.95), SIMDE_FLOAT32_C(    -8.66), SIMDE_FLOAT32_C(    86.71),
        SIMDE_FLOAT32_C(   -67.62), SIMDE_FLOAT32_C(   -80.76), SIMDE_FLOAT32_C(   -71.41), SIMDE_FLOAT32_C(   -26.70),
        SIMDE_FLOAT32_C(    73.96), SIMDE_FLOAT32_C(    61.35), SIMDE_FLOAT32_C(    50.17), SIMDE_FLOAT32_C(   -82.65) },
      { SIMDE_FLOAT32_C(    -1.64), SIMDE_FLOAT32_C(   -47.14), SIMDE_FLOAT32_C(    -6.17), SIMDE_FLOAT32_C(    94.46),
        SIMDE_FLOAT32_C(    83.87), SIMDE_FLOAT32_C(     9.63), SIMDE_FLOAT32_C(   -47.35), SIMDE_FLOAT32_C(   -69.77),
        SIMDE_FLOAT32_C(    58.45), SIMDE_FLOAT32_C(    98.46), SIMDE_FLOAT32_C(    18.36), SIMDE_FLOAT32_C(    37.30),
        SIMDE_FLOAT32_C(    65.61), SIMDE_FLOAT32_C(    18.67), SIMDE_FLOAT32_C(    21.98), SIMDE_FLOAT32_C(    87.18) },
      { SIMDE_FLOAT32_C( -4438.06), SIMDE_FLOAT32_C( -1868.13), SIMDE_FLOAT32_C(   144.33), SIMDE_FLOAT32_C(  1065.31),
        SIMDE_FLOAT32_C(  -435.03), SIMDE_FLOAT32_C(   422.96), SIMDE_FLOAT32_C(  -681.82), SIMDE_FLOAT32_C(  3695.12),
        SIMDE_FLOAT32_C( -3685.59), SIMDE_FLOAT32_C( -4231.76), SIMDE_FLOAT32_C(  3252.93), SIMDE_FLOAT32_C(  -354.23),
        SIMDE_FLOAT32_C( -5897.36), SIMDE_FLOAT32_C( -4138.94), SIMDE_FLOAT32_C(  4978.97), SIMDE_FLOAT32_C(  6912.45) } },
    { { SIMDE_FLOAT32_C(   -29.57), SIMDE_FLOAT32_C(    70.63), SIMDE_FLOAT32_C(    36.48), SIMDE_FLOAT32_C(   -19.61),
        SIMDE_FLOAT32_C(   -43.32), SIMDE_FLOAT32_C(   -72.18), SIMDE_FLOAT32_C(   -32.90), SIMDE_FLOAT32_C(   -10.94),
        SIMDE_FLOAT32_C(   -52.94), SIMDE_FLOAT32_C(    -4.31), SIMDE_FLOAT32_C(    62.36), SIMDE_FLOAT32_C(   -78.99),
        SIMDE_FLOAT32_C(   -42.96), SIMDE_FLOAT32_C(    12.54), SIMDE_FLOAT32_C(   -61.64), SIMDE_FLOAT32_C(    55.40) },
      { SIMDE_FLOAT32_C(    65.40), SIMDE_FLOAT32_C(    32.20), SIMDE_FLOAT32_C(    49.86), SIMDE_FLOAT32_C(    49.27),
        SIMDE_FLOAT32_C(   -58.17), SIMDE_FLOAT32_C(   -97.49), SIMDE_FLOAT32_C(    79.50), SIMDE_FLOAT32_C(   -99.71),
        SIMDE_FLOAT32_C(   -99.03), SIMDE_FLOAT32_C(    -2.14), SIMDE_FLOAT32_C(    37.59), SIMDE_FLOAT32_C(    66.58),
        SIMDE_FLOAT32_C(   -83.46), SIMDE_FLOAT32_C(   -40.43), SIMDE_FLOAT32_C(    53.75), SIMDE_FLOAT32_C(   -13.04) },
      { SIMDE_FLOAT32_C(   -69.80), SIMDE_FLOAT32_C(    -9.76), SIMDE_FLOAT32_C(    67.35), SIMDE_FLOAT32_C(   -13.12),
        SIMDE_FLOAT32_C(    18.06), SIMDE_FLOAT32_C(   -65.56), SIMDE_FLOAT32_C(    75.94), SIMDE_FLOAT32_C(    65.11),
        SIMDE_FLOAT32_C(    30.13), SIMDE_FLOAT32_C(    38.30), SIMDE_FLOAT32_C(    86.12), SIMDE_FLOAT32_C(    87.17),
        SIMDE_FLOAT32_C(   -49.16), SIMDE_FLOAT32_C(   -75.51), SIMDE_FLOAT32_C(    42.57), SIMDE_FLOAT32_C(   -83.76) },
      { SIMDE_FLOAT32_C(  2003.68), SIMDE_FLOAT32_C( -2264.53), SIMDE_FLOAT32_C( -1886.24), SIMDE_FLOAT32_C(   979.30),
        SIMDE_FLOAT32_C( -2537.98), SIMDE_FLOAT32_C( -6971.27), SIMDE_FLOAT32_C(  2539.61), SIMDE_FLOAT32_C( -1155.94),
        SIMDE_FLOAT32_C( -5272.78), SIMDE_FLOAT32_C(   -47.52), SIMDE_FLOAT32_C( -2430.23), SIMDE_FLOAT32_C(  5171.98),
        SIMDE_FLOAT32_C( -3536.28), SIMDE_FLOAT32_C(   582.50), SIMDE_FLOAT32_C(  3270.58), SIMDE_FLOAT32_C(   806.18) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 c = simde_mm512_loadu_ps(test_vec[i].c);
    simde__m512 r = simde_mm512_fnmsub_ps(a, b, c);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_fnmsub_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 c[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(    65.73), SIMDE_FLOAT64_C(    16.84), SIMDE_FLOAT64_C(   -62.22), SIMDE_FLOAT64_C(    41.28),
        SIMDE_FLOAT64_C(    41.92), SIMDE_FLOAT64_C(    45.60), SIMDE_FLOAT64_C(    20.93), SIMDE_FLOAT64_C(     8.26) },
      { SIMDE_FLOAT64_C(     7.90), SIMDE_FLOAT64_C(    31.90), SIMDE_FLOAT64_C(   -14.89), SIMDE_FLOAT64_C(   -98.38),
        SIMDE_FLOAT64_C(   -98.27), SIMDE_FLOAT64_C(    60.31), SIMDE_FLOAT64_C(   -39.17), SIMDE_FLOAT64_C(    82.51) },
      { SIMDE_FLOAT64_C(    35.14), SIMDE_FLOAT64_C(   -85.00), SIMDE_FLOAT64_C(   -10.86), SIMDE_FLOAT64_C(    33.90),
        SIMDE_FLOAT64_C(   -68.39), SIMDE_FLOAT64_C(   -87.95), SIMDE_FLOAT64_C(   -87.51), SIMDE_FLOAT64_C(   -78.91) },
      { SIMDE_FLOAT64_C(  -554.41), SIMDE_FLOAT64_C(  -452.20), SIMDE_FLOAT64_C(  -915.60), SIMDE_FLOAT64_C(  4027.23),
        SIMDE_FLOAT64_C(  4187.87), SIMDE_FLOAT64_C( -2662.19), SIMDE_FLOAT64_C(   907.34), SIMDE_FLOAT64_C(  -602.62) } },
    { { SIMDE_FLOAT64_C(   -48.41), SIMDE_FLOAT64_C(    52.12), SIMDE_FLOAT64_C(   -76.82), SIMDE_FLOAT64_C(   -20.81),
        SIMDE_FLOAT64_C(    40.27), SIMDE_FLOAT64_C(   -85.67), SIMDE_FLOAT64_C(   -20.01), SIMDE_FLOAT64_C(     6.00) },
      { SIMDE_FLOAT64_C(    31.17), SIMDE_FLOAT64_C(    17.77), SIMDE_FLOAT64_C(   -52.72), SIMDE_FLOAT64_C(   -26.91),
        SIMDE_FLOAT64_C(   -36.63), SIMDE_FLOAT64_C(    68.22), SIMDE_FLOAT64_C(    81.34), SIMDE_FLOAT64_C(    71.27) },
      { SIMDE_FLOAT64_C(     0.11), SIMDE_FLOAT64_C(   -33.55), SIMDE_FLOAT64_C(    72.89), SIMDE_FLOAT64_C(     1.84),
        SIMDE_FLOAT64_C(   -73.23), SIMDE_FLOAT64_C(   -66.27), SIMDE_FLOAT64_C(   -15.65), SIMDE_FLOAT64_C(    61.91) },
      { SIMDE_FLOAT64_C(  1508.83), SIMDE_FLOAT64_C(  -892.62), SIMDE_FLOAT64_C( -4122.84), SIMDE_FLOAT64_C(  -561.84),
        SIMDE_FLOAT64_C(  1548.32), SIMDE_FLOAT64_C(  5910.68), SIMDE_FLOAT64_C(  1643.26), SIMDE_FLOAT64_C(  -489.53) } },
    { { SIMDE_FLOAT64_C(   -51.27), SIMDE_FLOAT64_C(    73.48), SIMDE_FLOAT64_C(    -4.19), SIMDE_FLOAT64_C(   -19.66),
        SIMDE_FLOAT64_C(    85.53), SIMDE_FLOAT64_C(     8.30), SIMDE_FLOAT64_C(     1.43), SIMDE_FLOAT64_C(   -62.88) },
      { SIMDE_FLOAT64_C(   -39.58), SIMDE_FLOAT64_C(    24.60), SIMDE_FLOAT64_C(    16.31), SIMDE_FLOAT64_C(   -99.31),
        SIMDE_FLOAT64_C(    38.94), SIMDE_FLOAT64_C(    96.30), SIMDE_FLOAT64_C(     6.69), SIMDE_FLOAT64_C(   -29.89) },
      { SIMDE_FLOAT64_C(    14.08), SIMDE_FLOAT64_C(    53.98), SIMDE_FLOAT64_C(    43.19), SIMDE_FLOAT64_C(    77.45),
        SIMDE_FLOAT64_C(    22.19), SIMDE_FLOAT64_C(    24.54), SIMDE_FLOAT64_C(    48.72), SIMDE_FLOAT64_C(   -77.69) },
      { SIMDE_FLOAT64_C( -2043.35), SIMDE_FLOAT64_C( -1861.59), SIMDE_FLOAT64_C(    25.15), SIMDE_FLOAT64_C( -2029.88),
        SIMDE_FLOAT64_C( -3352.73), SIMDE_FLOAT64_C(  -823.83), SIMDE_FLOAT64_C(   -58.29), SIMDE_FLOAT64_C( -1801.79) } },
    { { SIMDE_FLOAT64_C(    90.99), SIMDE_FLOAT64_C(    21.61), SIMDE_FLOAT64_C(    24.15), SIMDE_FLOAT64_C(   -82.24),
        SIMDE_FLOAT64_C(    55.34), SIMDE_FLOAT64_C(   -91.51), SIMDE_FLOAT64_C(    79.67), SIMDE_FLOAT64_C(   -95.94) },
      { SIMDE_FLOAT64_C(    81.97), SIMDE_FLOAT64_C(   -24.52), SIMDE_FLOAT64_C(   -15.60), SIMDE_FLOAT64_C(    67.51),
        SIMDE_FLOAT64_C(    83.77), SIMDE_FLOAT64_C(    85.83), SIMDE_FLOAT64_C(   -95.37), SIMDE_FLOAT64_C(   -55.81) },
      { SIMDE_FLOAT64_C(    10.43), SIMDE_FLOAT64_C(    20.94), SIMDE_FLOAT64_C(   -55.12), SIMDE_FLOAT64_C(   -50.63),
        SIMDE_FLOAT64_C(    17.24), SIMDE_FLOAT64_C(    51.57), SIMDE_FLOAT64_C(    19.47), SIMDE_FLOAT64_C(   -68.68) },
      { SIMDE_FLOAT64_C( -7468.88), SIMDE_FLOAT64_C(   508.94), SIMDE_FLOAT64_C(   431.86), SIMDE_FLOAT64_C(  5602.65),
        SIMDE_FLOAT64_C( -4653.07), SIMDE_FLOAT64_C(  7802.73), SIMDE_FLOAT64_C(  7578.66), SIMDE_FLOAT64_C( -5285.73) } },
    { { SIMDE_FLOAT64_C(     5.55), SIMDE_FLOAT64_C(   -37.33), SIMDE_FLOAT64_C(   -91.23), SIMDE_FLOAT64_C(   -72.26),
        SIMDE_FLOAT64_C(    87.21), SIMDE_FLOAT64_C(    57.48), SIMDE_FLOAT64_C(   -49.95), SIMDE_FLOAT64_C(    78.20) },
      { SIMDE_FLOAT64_C(   -20.90), SIMDE_FLOAT64_C(    74.20), SIMDE_FLOAT64_C(    95.96), SIMDE_FLOAT64_C(   -65.57),
        SIMDE_FLOAT64_C(    82.69), SIMDE_FLOAT64_C(    75.63), SIMDE_FLOAT64_C(   -61.50), SIMDE_FLOAT64_C(    64.66) },
      { SIMDE_FLOAT64_C(   -48.89), SIMDE_FLOAT64_C(    22.90), SIMDE_FLOAT64_C(    32.17), SIMDE_FLOAT64_C(   -65.12),
        SIMDE_FLOAT64_C(     8.73), SIMDE_FLOAT64_C(    36.80), SIMDE_FLOAT64_C(   -20.93), SIMDE_FLOAT64_C(   -80.84) },
      { SIMDE_FLOAT64_C(   164.88), SIMDE_FLOAT64_C(  2746.99), SIMDE_FLOAT64_C(  8722.26), SIMDE_FLOAT64_C( -4672.97),
        SIMDE_FLOAT64_C( -7220.12), SIMDE_FLOAT64_C( -4384.01), SIMDE_FLOAT64_C( -3051.00), SIMDE_FLOAT64_C( -4975.57) } },
    { { SIMDE_FLOAT64_C(   -42.25), SIMDE_FLOAT64_C(    23.95), SIMDE_FLOAT64_C(   -31.47), SIMDE_FLOAT64_C(    74.99),
        SIMDE_FLOAT64_C(   -24.48), SIMDE_FLOAT64_C(    88.00), SIMDE_FLOAT64_C(   -93.69), SIMDE_FLOAT64_C(    81.07) },
      { SIMDE_FLOAT64_C(   -49.33), SIMDE_FLOAT64_C(   -84.92), SIMDE_FLOAT64_C(   -91.19), SIMDE_FLOAT64_C(   -62.12),
        SIMDE_FLOAT64_C(    72.56), SIMDE_FLOAT64_C(   -41.14), SIMDE_FLOAT64_C(   -83.92), SIMDE_FLOAT64_C(   -48.34) },
      { SIMDE_FLOAT64_C(   -66.94), SIMDE_FLOAT64_C(   -87.96), SIMDE_FLOAT64_C(   -13.91), SIMDE_FLOAT64_C(   -84.25),
        SIMDE_FLOAT64_C(    87.67), SIMDE_FLOAT64_C(    24.58), SIMDE_FLOAT64_C(    80.41), SIMDE_FLOAT64_C(   -61.22) },
      { SIMDE_FLOAT64_C( -2017.25), SIMDE_FLOAT64_C(  2121.79), SIMDE_FLOAT64_C( -2855.84), SIMDE_FLOAT64_C(  4742.63),
        SIMDE_FLOAT64_C(  1688.60), SIMDE_FLOAT64_C(  3595.74), SIMDE_FLOAT64_C( -7942.87), SIMDE_FLOAT64_C(  3980.14) } },
    { { SIMDE_FLOAT64_C(   -52.52), SIMDE_FLOAT64_C(    12.58), SIMDE_FLOAT64_C(   -26.34), SIMDE_FLOAT64_C(    56.21),
        SIMDE_FLOAT64_C(   -50.62), SIMDE_FLOAT64_C(    52.73), SIMDE_FLOAT64_C(    75.37), SIMDE_FLOAT64_C(     7.13) },
      { SIMDE_FLOAT64_C(   -23.32), SIMDE_FLOAT64_C(   -56.10), SIMDE_FLOAT64_C(   -17.88), SIMDE_FLOAT64_C(    52.20),
        SIMDE_FLOAT64_C(   -68.10), SIMDE_FLOAT64_C(   -11.57), SIMDE_FLOAT64_C(    33.27), SIMDE_FLOAT64_C(   -17.43) },
      { SIMDE_FLOAT64_C(     3.50), SIMDE_FLOAT64_C(    42.08), SIMDE_FLOAT64_C(    20.45), SIMDE_FLOAT64_C(   -23.94),
        SIMDE_FLOAT64_C(   -99.05), SIMDE_FLOAT64_C(    36.53), SIMDE_FLOAT64_C(    27.72), SIMDE_FLOAT64_C(   -66.00) },
      { SIMDE_FLOAT64_C( -1228.27), SIMDE_FLOAT64_C(   663.66), SIMDE_FLOAT64_C(  -491.41), SIMDE_FLOAT64_C( -2910.22),
        SIMDE_FLOAT64_C( -3348.17), SIMDE_FLOAT64_C(   573.56), SIMDE_FLOAT64_C( -2535.28), SIMDE_FLOAT64_C(   190.28) } },
    { { SIMDE_FLOAT64_C(    48.57), SIMDE_FLOAT64_C(   -86.20), SIMDE_FLOAT64_C(   -50.25), SIMDE_FLOAT64_C(    36.24),
        SIMDE_FLOAT64_C(    38.39), SIMDE_FLOAT64_C(   -69.84), SIMDE_FLOAT64_C(    75.01), SIMDE_FLOAT64_C(    85.87) },
      { SIMDE_FLOAT64_C(    42.73), SIMDE_FLOAT64_C(   -51.33), SIMDE_FLOAT64_C(    42.08), SIMDE_FLOAT64_C(    92.12),
        SIMDE_FLOAT64_C(   -98.60), SIMDE_FLOAT64_C(    17.44), SIMDE_FLOAT64_C(    -0.76), SIMDE_FLOAT64_C(   -21.92) },
      { SIMDE_FLOAT64_C(    61.34), SIMDE_FLOAT64_C(    81.36), SIMDE_FLOAT64_C(   -69.73), SIMDE_FLOAT64_C(    93.24),
        SIMDE_FLOAT64_C(   -30.21), SIMDE_FLOAT64_C(    63.55), SIMDE_FLOAT64_C(   -24.19), SIMDE_FLOAT64_C(    73.29) },
      { SIMDE_FLOAT64_C( -2136.74), SIMDE_FLOAT64_C( -4506.01), SIMDE_FLOAT64_C(  2184.25), SIMDE_FLOAT64_C( -3431.67),
        SIMDE_FLOAT64_C(  3815.46), SIMDE_FLOAT64_C(  1154.46), SIMDE_FLOAT64_C(    81.20), SIMDE_FLOAT64_C(  1808.98) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d c = simde_mm512_loadu_pd(test_vec[i].c);
    simde__m512d r = simde_mm512_fnmsub_pd(a, b, c);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fnmsub_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fnmsub_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
