// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type release struct {
	common

	forgejoRelease *forgejo_sdk.Release
}

var _ f3_tree.ForgeDriverInterface = &release{}

func newRelease() generic.NodeDriverInterface {
	return &release{}
}

func (o *release) SetNative(release any) {
	o.forgejoRelease = release.(*forgejo_sdk.Release)
}

func (o *release) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoRelease.ID)
}

func (o *release) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *release) ToFormat() f3.Interface {
	if o.forgejoRelease == nil {
		return o.NewFormat()
	}

	return &f3.Release{
		Common:          f3.NewCommon(o.GetNativeID()),
		TagName:         o.forgejoRelease.TagName,
		TargetCommitish: o.forgejoRelease.Target,
		Name:            o.forgejoRelease.Title,
		Body:            o.forgejoRelease.Note,
		Draft:           o.forgejoRelease.IsDraft,
		Prerelease:      o.forgejoRelease.IsPrerelease,
		PublisherID:     f3_tree.NewUserReference(o.forgejoRelease.Publisher.ID),
		Created:         o.forgejoRelease.CreatedAt,
	}
}

func (o *release) FromFormat(content f3.Interface) {
	release := content.(*f3.Release)
	o.forgejoRelease = &forgejo_sdk.Release{
		ID:           util.ParseInt(release.GetID()),
		TagName:      release.TagName,
		Target:       release.TargetCommitish,
		Title:        release.Name,
		Note:         release.Body,
		IsDraft:      release.Draft,
		IsPrerelease: release.Prerelease,
		Publisher: &forgejo_sdk.User{
			ID: release.PublisherID.GetIDAsInt(),
		},
		CreatedAt: release.Created,
	}
}

func (o *release) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	release, resp, err := o.getClient().GetRelease(owner, project, node.GetID().Int64())
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("release %v %w", o, err))
	}
	o.forgejoRelease = release
	return true
}

func (o *release) Put(ctx context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	o.maybeSudoID(ctx, o.forgejoRelease.Publisher.ID)
	defer o.notSudo()

	release, _, err := o.getClient().CreateRelease(owner, project, forgejo_sdk.CreateReleaseOption{
		TagName:      o.forgejoRelease.TagName,
		Target:       o.forgejoRelease.Target,
		Title:        o.forgejoRelease.Title,
		Note:         o.forgejoRelease.Note,
		IsDraft:      o.forgejoRelease.IsDraft,
		IsPrerelease: o.forgejoRelease.IsPrerelease,
	})
	if err != nil {
		panic(err)
	}
	o.forgejoRelease = release
	return id.NewNodeID(o.GetNativeID())
}

func (o *release) Patch(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, _, err := o.getClient().EditRelease(owner, project, node.GetID().Int64(), forgejo_sdk.EditReleaseOption{
		TagName:      o.forgejoRelease.TagName,
		Target:       o.forgejoRelease.Target,
		Title:        o.forgejoRelease.Title,
		Note:         o.forgejoRelease.Note,
		IsDraft:      &o.forgejoRelease.IsDraft,
		IsPrerelease: &o.forgejoRelease.IsPrerelease,
	})
	if err != nil {
		panic(err)
	}
}

func (o *release) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, err := o.getClient().DeleteRelease(owner, project, node.GetID().Int64())
	if err != nil {
		panic(err)
	}
}
