/*
 * QDM2 compatible decoder
 * Copyright (c) 2003 Ewald Snel
 * Copyright (c) 2005 Benjamin Larsson
 * Copyright (c) 2005 Alex Beregszaszi
 * Copyright (c) 2005 Roberto Togni
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

 /**
 * @file
 * Various QDM2 tables.
 */

#ifndef AVCODEC_QDM2DATA_H
#define AVCODEC_QDM2DATA_H

#include <stdint.h>

/** VLC TABLES **/

/* values in this table range from -1..23; adjust retrieved value by -1 */
static const uint8_t tab_level[24][2] = {
    { 12,  4 }, { 17,  4 }, {  1,  6 }, {  8,  6 }, {  9,  5 }, { 20,  7 },
    {  3,  7 }, {  5,  6 }, {  6,  6 }, {  2,  7 }, { 22,  9 }, { 23, 10 },
    {  0, 10 }, { 21,  8 }, { 11,  4 }, { 19,  5 }, {  7,  6 }, {  4,  6 },
    { 16,  3 }, { 10,  4 }, { 18,  4 }, { 15,  3 }, { 13,  3 }, { 14,  3 },
};

/* values in this table range from -1..36; adjust retrieved value by -1 */
static const uint8_t tab_diff[33][2] = {
    {  2,  3 }, {  1,  3 }, {  5,  3 }, { 14,  8 }, { 20,  9 }, { 26, 10 },
    { 25, 12 }, { 32, 12 }, { 19, 11 }, { 16,  8 }, { 24,  9 }, { 17,  9 },
    { 12,  7 }, { 13,  7 }, {  9,  5 }, {  7,  4 }, {  3,  2 }, {  4,  3 },
    {  8,  6 }, { 11,  6 }, { 18,  8 }, { 15,  8 }, { 30, 11 }, { 36, 13 },
    { 34, 13 }, { 29, 13 }, {  0, 13 }, { 21, 10 }, { 28, 10 }, { 23, 10 },
    { 22,  8 }, { 10,  6 }, {  6,  4 },
};

/* values in this table range from -1..5; adjust retrieved value by -1 */
static const uint8_t tab_run[6][2] = {
    { 1, 1 }, { 2, 2 }, { 3, 3 }, { 4, 4 }, { 5, 5 }, { 0, 5 },
};

/* values in this table range from -1..19; adjust retrieved value by -1 */
static const uint8_t tab_tone_level_idx_hi1[20][2] = {
    {  4,  3 }, {  5,  5 }, {  9, 10 }, { 11, 11 }, { 13, 12 }, { 14, 12 },
    { 10, 10 }, { 12, 11 }, { 17, 14 }, { 16, 14 }, { 18, 15 }, {  0, 15 },
    { 19, 14 }, { 15, 12 }, {  8,  8 }, {  7,  7 }, {  6,  6 }, {  1,  4 },
    {  2,  2 }, {  3,  1 },
};

/* values in this table range from -1..23; adjust retrieved value by -1 */
static const uint8_t tab_tone_level_idx_mid[13][2] = {
    { 18,  2 }, { 19,  4 }, { 20,  6 }, { 14,  7 }, { 21,  8 }, { 13,  9 },
    { 22, 10 }, { 12, 11 }, { 23, 12 }, {  0, 12 }, { 15,  5 }, { 16,  3 },
    { 17,  1 },
};

/* values in this table range from -1..23; adjust retrieved value by -1 */
static const uint8_t tab_tone_level_idx_hi2[18][2] = {
    { 14,  4 }, { 11,  6 }, { 19,  7 }, {  9,  7 }, { 13,  5 }, { 10,  6 },
    { 20,  8 }, {  8,  8 }, {  6, 10 }, { 23, 11 }, {  0, 11 }, { 21,  9 },
    {  7,  8 }, { 12,  5 }, { 18,  4 }, { 16,  2 }, { 15,  2 }, { 17,  2 },
};

/* values in this table range from -1..8; adjust retrieved value by -1 */
static const uint8_t tab_type30[9][2] = {
    { 2, 3 }, { 6, 4 }, { 7, 5 }, { 8, 6 }, { 0, 6 }, { 5, 3 }, { 1, 3 },
    { 3, 2 }, { 4, 2 },
};

/* values in this table range from -1..9; adjust retrieved value by -1 */
static const uint8_t tab_type34[10][2] = {
    { 1, 4 }, { 9, 5 }, { 0, 5 }, { 3, 3 }, { 7, 3 }, { 8, 3 }, { 2, 3 },
    { 4, 3 }, { 6, 3 }, { 5, 3 },
};

static const uint8_t tab_fft_tone_offset_sizes[] = {
    23, 28, 31, 34, 37
};

static const uint8_t tab_fft_tone_offset[][2] = {
    /* First table - 23 entries with range -1..21 */
    {  2,  2 }, {  7,  7 }, { 15,  8 }, { 21,  8 }, {  3,  6 }, {  6,  6 },
    { 13,  7 }, { 14,  8 }, { 18,  8 }, {  4,  4 }, {  5,  5 }, { 11,  7 },
    { 10,  7 }, { 20,  6 }, { 12,  8 }, { 16,  9 }, { 22, 10 }, {  0, 10 },
    { 17,  7 }, { 19,  6 }, {  8,  6 }, {  9,  6 }, {  1,  1 },
    /* Second table - 28 entries with range -1..26 */
    {  8,  6 }, {  2,  6 }, {  7,  6 }, { 23,  7 }, { 12,  7 }, {  5,  4 },
    { 10,  6 }, { 20,  8 }, { 25,  9 }, { 26, 10 }, { 27, 11 }, {  0, 11 },
    { 22,  7 }, {  9,  5 }, { 13,  6 }, { 17,  6 }, {  4,  5 }, { 14,  6 },
    { 19,  7 }, { 24,  7 }, {  3,  6 }, { 11,  6 }, { 21,  6 }, { 18,  6 },
    { 16,  6 }, { 15,  6 }, {  6,  3 }, {  1,  1 },
    /* Third table - 31 entries with range -1..30  (1 omitted) */
    { 14,  7 }, { 17,  7 }, { 15,  7 }, { 23,  9 }, { 28, 10 }, { 29, 11 },
    { 30, 13 }, {  0, 13 }, { 31, 12 }, { 25,  8 }, { 10,  5 }, {  8,  4 },
    {  9,  4 }, {  4,  4 }, { 22,  8 }, {  3,  8 }, { 21,  8 }, { 26,  9 },
    { 27,  9 }, { 12,  6 }, { 11,  5 }, { 16,  7 }, { 18,  7 }, { 20,  8 },
    { 24,  8 }, { 19,  7 }, { 13,  5 }, {  5,  3 }, {  1,  2 }, {  6,  3 },
    {  7,  3 },
    /* Fourth table - 34 entries with range -1..33 (1 omitted) */
    {  4,  4 }, {  7,  4 }, { 10,  4 }, {  3, 10 }, { 27, 10 }, { 29, 10 },
    { 28, 10 }, { 22,  8 }, { 21,  7 }, { 15,  6 }, { 14,  5 }, {  8,  4 },
    { 16,  6 }, { 19,  7 }, { 23,  8 }, { 26,  9 }, { 30, 10 }, { 33, 13 },
    { 34, 14 }, {  0, 14 }, { 32, 12 }, { 31, 11 }, { 12,  5 }, {  5,  3 },
    {  9,  3 }, {  1,  4 }, { 20,  7 }, { 25,  8 }, { 24,  8 }, { 18,  6 },
    { 17,  5 }, {  6,  3 }, { 11,  4 }, { 13,  4 },
    /* Fifth table - 37 entries with range -1..36 (1 omitted) */
    {  5,  3 }, {  4,  3 }, { 19,  8 }, { 33, 12 }, { 31, 12 }, { 28, 11 },
    { 34, 14 }, { 37, 14 }, { 35, 15 }, {  0, 15 }, { 36, 14 }, { 32, 12 },
    { 30, 11 }, { 24,  9 }, { 22,  8 }, { 23,  9 }, { 29, 10 }, { 27, 10 },
    { 17,  6 }, { 14,  5 }, {  7,  4 }, { 12,  5 }, {  1,  6 }, { 26,  9 },
    {  3,  9 }, { 25,  8 }, { 20,  7 }, {  8,  4 }, { 10,  4 }, { 13,  4 },
    { 15,  6 }, { 16,  6 }, { 18,  6 }, { 21,  6 }, { 11,  4 }, {  9,  3 },
    {  6,  3 },
};

/** FFT TABLES **/

/* values in this table range from -1..27; adjust retrieved value by -1 */
static const uint8_t fft_level_exp_alt[28][2] = {
    { 18,  3 }, { 16,  3 }, { 22,  7 }, {  8, 10 }, {  4, 10 }, {  3,  9 },
    {  2,  8 }, { 23,  8 }, { 10,  8 }, { 11,  7 }, { 21,  5 }, { 20,  4 },
    {  1,  7 }, {  7, 10 }, {  5, 10 }, {  9,  9 }, {  6, 10 }, { 25, 11 },
    { 26, 12 }, { 27, 13 }, {  0, 13 }, { 24,  9 }, { 12,  6 }, { 13,  5 },
    { 14,  4 }, { 19,  3 }, { 15,  3 }, { 17,  2 },
};

/* values in this table range from -1..19; adjust retrieved value by -1 */
static const uint8_t fft_level_exp[20][2] = {
    {  3,  3 }, { 11,  6 }, { 16,  9 }, { 17, 10 }, { 18, 11 }, { 19, 12 },
    {  0, 12 }, { 15,  8 }, { 14,  7 }, {  9,  5 }, {  7,  4 }, {  2,  3 },
    {  4,  3 }, {  1,  3 }, {  5,  3 }, { 12,  6 }, { 13,  6 }, { 10,  5 },
    {  8,  4 }, {  6,  3 },
};

/* values in this table range from -1..6; adjust retrieved value by -1 */
static const uint8_t fft_stereo_exp[7][2] = {
    { 2, 2 }, { 3, 3 }, { 4, 4 }, { 5, 5 }, { 6, 6 }, { 0, 6 }, { 1, 1 },
};

/* values in this table range from -1..8; adjust retrieved value by -1 */
static const uint8_t fft_stereo_phase[9][2] = {
    { 2, 2 }, { 1, 2 }, { 3, 4 }, { 7, 4 }, { 6, 5 }, { 5, 6 }, { 0, 6 },
    { 4, 4 }, { 8, 2 },
};

static const int fft_cutoff_index_table[4][2] = {
    { 1, 2 }, {-1, 0 }, {-1,-2 }, { 0, 0 }
};

static const int16_t fft_level_index_table[256] = {
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
};

static const uint8_t last_coeff[3] = {
    4, 7, 10
};

static const uint8_t coeff_per_sb_for_avg[3][30] = {
    { 0, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3 },
    { 0, 1, 2, 2, 3, 3, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6 },
    { 0, 1, 2, 3, 4, 4, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9 }
};

static const uint32_t dequant_table[3][10][30] = {
    { { 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 256, 256, 205, 154, 102, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 51, 102, 154, 205, 256, 238, 219, 201, 183, 165, 146, 128, 110, 91, 73, 55, 37, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 18, 37, 55, 73, 91, 110, 128, 146, 165, 183, 201, 219, 238, 256, 228, 199, 171, 142, 114, 85, 57, 28 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } },
    { { 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 256, 171, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 85, 171, 256, 171, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 85, 171, 256, 219, 183, 146, 110, 73, 37, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 73, 110, 146, 183, 219, 256, 228, 199, 171, 142, 114, 85, 57, 28, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 28, 57, 85, 114, 142, 171, 199, 228, 256, 213, 171, 128, 85, 43 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } },
    { { 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 256, 256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 256, 171, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 85, 171, 256, 192, 128, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 128, 192, 256, 205, 154, 102, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 51, 102, 154, 205, 256, 213, 171, 128, 85, 43, 0, 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 43, 85, 128, 171, 213, 256, 213, 171, 128, 85, 43 } }
};

static const uint8_t coeff_per_sb_for_dequant[3][30] = {
    { 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3 },
    { 0, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6 },
    { 0, 1, 2, 3, 4, 4, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9 }
};

/* first index is subband, 2nd index is 0, 1 or 3 (2 is unused) */
static const int8_t tone_level_idx_offset_table[30][4] = {
    { -50, -50,  0, -50 },
    { -50, -50,  0, -50 },
    { -50,  -9,  0, -19 },
    { -16,  -6,  0, -12 },
    { -11,  -4,  0,  -8 },
    {  -8,  -3,  0,  -6 },
    {  -7,  -3,  0,  -5 },
    {  -6,  -2,  0,  -4 },
    {  -5,  -2,  0,  -3 },
    {  -4,  -1,  0,  -3 },
    {  -4,  -1,  0,  -2 },
    {  -3,  -1,  0,  -2 },
    {  -3,  -1,  0,  -2 },
    {  -3,  -1,  0,  -2 },
    {  -2,  -1,  0,  -1 },
    {  -2,  -1,  0,  -1 },
    {  -2,  -1,  0,  -1 },
    {  -2,   0,  0,  -1 },
    {  -2,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,  -1 },
    {  -1,   0,  0,   0 },
    {  -1,   0,  0,   0 },
    {  -1,   0,  0,   0 },
    {  -1,   0,  0,   0 }
};

/* all my samples have 1st index 0 or 1 */
/* second index is subband, only indexes 0-29 seem to be used */
static const int8_t coding_method_table[5][30] = {
    { 34, 30, 24, 24, 16, 16, 16, 16, 10, 10, 10, 10, 10, 10, 10,
      10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10
    },
    { 34, 30, 24, 24, 16, 16, 16, 16, 10, 10, 10, 10, 10, 10, 10,
      10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10
    },
    { 34, 30, 30, 30, 24, 24, 16, 16, 16, 16, 16, 16, 10, 10, 10,
      10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10
    },
    { 34, 34, 30, 30, 24, 24, 24, 24, 16, 16, 16, 16, 16, 16, 16,
      16, 16, 16, 16, 16, 16, 16, 10, 10, 10, 10, 10, 10, 10, 10
    },
    { 34, 34, 30, 30, 30, 30, 30, 30, 24, 24, 24, 24, 24, 24, 24,
      24, 24, 24, 24, 24, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16
    },
};

static const int vlc_stage3_values[60] = {
        0,     1,     2,     3,     4,     6,     8,    10,    12,    16,    20,    24,
       28,    36,    44,    52,    60,    76,    92,   108,   124,   156,   188,   220,
      252,   316,   380,   444,   508,   636,   764,   892,  1020,  1276,  1532,  1788,
     2044,  2556,  3068,  3580,  4092,  5116,  6140,  7164,  8188, 10236, 12284, 14332,
    16380, 20476, 24572, 28668, 32764, 40956, 49148, 57340, 65532, 81916, 98300,114684
};

static const float fft_tone_sample_table[4][16][5] = {
    { { .0100000000f,-.0037037037f,-.0020000000f,-.0069444444f,-.0018416207f },
      { .0416666667f, .0000000000f, .0000000000f,-.0208333333f,-.0123456791f },
      { .1250000000f, .0558035709f, .0330687836f,-.0164473690f,-.0097465888f },
      { .1562500000f, .0625000000f, .0370370370f,-.0062500000f,-.0037037037f },
      { .1996007860f, .0781250000f, .0462962948f, .0022727272f, .0013468013f },
      { .2000000000f, .0625000000f, .0370370373f, .0208333333f, .0074074073f },
      { .2127659619f, .0555555556f, .0329218097f, .0208333333f, .0123456791f },
      { .2173913121f, .0473484844f, .0280583613f, .0347222239f, .0205761325f },
      { .2173913121f, .0347222239f, .0205761325f, .0473484844f, .0280583613f },
      { .2127659619f, .0208333333f, .0123456791f, .0555555556f, .0329218097f },
      { .2000000000f, .0208333333f, .0074074073f, .0625000000f, .0370370370f },
      { .1996007860f, .0022727272f, .0013468013f, .0781250000f, .0462962948f },
      { .1562500000f,-.0062500000f,-.0037037037f, .0625000000f, .0370370370f },
      { .1250000000f,-.0164473690f,-.0097465888f, .0558035709f, .0330687836f },
      { .0416666667f,-.0208333333f,-.0123456791f, .0000000000f, .0000000000f },
      { .0100000000f,-.0069444444f,-.0018416207f,-.0037037037f,-.0020000000f } },

    { { .0050000000f,-.0200000000f, .0125000000f,-.3030303030f, .0020000000f },
      { .1041666642f, .0400000000f,-.0250000000f, .0333333333f,-.0200000000f },
      { .1250000000f, .0100000000f, .0142857144f,-.0500000007f,-.0200000000f },
      { .1562500000f,-.0006250000f,-.00049382716f,-.000625000f,-.00049382716f },
      { .1562500000f,-.0006250000f,-.00049382716f,-.000625000f,-.00049382716f },
      { .1250000000f,-.0500000000f,-.0200000000f, .0100000000f, .0142857144f },
      { .1041666667f, .0333333333f,-.0200000000f, .0400000000f,-.0250000000f },
      { .0050000000f,-.3030303030f, .0020000001f,-.0200000000f, .0125000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f } },

    { { .1428571492f, .1250000000f,-.0285714287f,-.0357142873f, .0208333333f },
      { .1818181818f, .0588235296f, .0333333333f, .0212765951f, .0100000000f },
      { .1818181818f, .0212765951f, .0100000000f, .0588235296f, .0333333333f },
      { .1428571492f,-.0357142873f, .0208333333f, .1250000000f,-.0285714287f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f } },

    { { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f },
      { .0000000000f, .0000000000f, .0000000000f, .0000000000f, .0000000000f } }
};

static const float fft_tone_level_table[2][64] = { {
/* pow ~ (i > 46) ? 0 : (((((i & 1) ? 431 : 304) << (i >> 1))) / 1024.0); */
    0.17677669f, 0.42677650f, 0.60355347f, 0.85355347f,
    1.20710683f, 1.68359375f, 2.37500000f, 3.36718750f,
    4.75000000f, 6.73437500f, 9.50000000f, 13.4687500f,
    19.0000000f, 26.9375000f, 38.0000000f, 53.8750000f,
    76.0000000f, 107.750000f, 152.000000f, 215.500000f,
    304.000000f, 431.000000f, 608.000000f, 862.000000f,
    1216.00000f, 1724.00000f, 2432.00000f, 3448.00000f,
    4864.00000f, 6896.00000f, 9728.00000f, 13792.0000f,
    19456.0000f, 27584.0000f, 38912.0000f, 55168.0000f,
    77824.0000f, 110336.000f, 155648.000f, 220672.000f,
    311296.000f, 441344.000f, 622592.000f, 882688.000f,
    1245184.00f, 1765376.00f, 2490368.00f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
  }, {
/* pow = (i > 45) ? 0 : ((((i & 1) ? 431 : 304) << (i >> 1)) / 512.0); */
    0.59375000f, 0.84179688f, 1.18750000f, 1.68359375f,
    2.37500000f, 3.36718750f, 4.75000000f, 6.73437500f,
    9.50000000f, 13.4687500f, 19.0000000f, 26.9375000f,
    38.0000000f, 53.8750000f, 76.0000000f, 107.750000f,
    152.000000f, 215.500000f, 304.000000f, 431.000000f,
    608.000000f, 862.000000f, 1216.00000f, 1724.00000f,
    2432.00000f, 3448.00000f, 4864.00000f, 6896.00000f,
    9728.00000f, 13792.0000f, 19456.0000f, 27584.0000f,
    38912.0000f, 55168.0000f, 77824.0000f, 110336.000f,
    155648.000f, 220672.000f, 311296.000f, 441344.000f,
    622592.000f, 882688.000f, 1245184.00f, 1765376.00f,
    2490368.00f, 3530752.00f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f,
    0.00000000f, 0.00000000f, 0.00000000f, 0.00000000f
} };

static const float fft_tone_envelope_table[4][31] = {
    { .009607375f, .038060248f, .084265202f, .146446645f, .222214907f, .308658302f,
      .402454883f, .500000060f, .597545207f, .691341758f, .777785182f, .853553414f,
      .915734828f, .961939812f, .990392685f, 1.00000000f, .990392625f, .961939752f,
      .915734768f, .853553295f, .777785063f, .691341639f, .597545087f, .500000000f,
      .402454853f, .308658272f, .222214878f, .146446615f, .084265172f, .038060218f,
      .009607345f },
    { .038060248f, .146446645f, .308658302f, .500000060f, .691341758f, .853553414f,
      .961939812f, 1.00000000f, .961939752f, .853553295f, .691341639f, .500000000f,
      .308658272f, .146446615f, .038060218f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f },
    { .146446645f, .500000060f, .853553414f, 1.00000000f, .853553295f, .500000000f,
      .146446615f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f },
    { .500000060f, 1.00000000f, .500000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f, .000000000f, .000000000f, .000000000f, .000000000f, .000000000f,
      .000000000f }
};

static const float sb_noise_attenuation[32] = {
    0.0f, 0.0f, 0.3f, 0.4f, 0.5f, 0.7f, 1.0f, 1.0f,
    1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
    1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
    1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
};

static const uint8_t fft_subpackets[32] = {
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0
};

/* first index is joined_stereo, second index is 0 or 2 (1 is unused) */
static const float dequant_1bit[2][3] = {
    {-0.920000f, 0.000000f, 0.920000f },
    {-0.890000f, 0.000000f, 0.890000f }
};

static const float type30_dequant[8] = {
   -1.0f,-0.625f,-0.291666656732559f,0.0f,
   0.25f,0.5f,0.75f,1.0f,
};

static const float type34_delta[10] = { // FIXME: covers 8 entries..
    -1.0f,-0.60947573184967f,-0.333333343267441f,-0.138071194291115f,0.0f,
    0.138071194291115f,0.333333343267441f,0.60947573184967f,1.0f,0.0f,
};

#endif /* AVCODEC_QDM2DATA_H */
